class LandingPage {
  constructor(config) {
    this.config = config;
    this.reviews = config.reviews || {};
    this.coupons = config.coupons || {};

    document.addEventListener("DOMContentLoaded", () => {
      this.init();
    });
  }

  init() {
    const results = {
      handleServiceBodyClasses: this.handleServiceBodyClasses(),
      handleModuleClasses: this.handleModuleClasses(),
      setFormValues: this.setFormValues(),
      handleFormContainer: this.handleFormContainer(),
      replacePhoneCTAs: this.replacePhoneCTAs(),
      replacePlaceholders: this.replacePlaceholders(),
      handleFinanceModule: this.handleFinanceModule(),
      handleFDICLogo: this.handleFDICLogo(),
      injectReviews: this.injectReviews(),
      injectCoupons: this.injectCoupons(),
      setupTooltips: this.setupTooltips(),
      setupMenuToggle: this.setupMenuToggle(),
      equalizeCouponHeights: this.equalizeCouponHeights(),
      updateExpirationDates: this.updateExpirationDates(),
    };

    const failed = Object.entries(results).filter(([, success]) => !success);
    if (failed.length === 0) {
      console.log("All LandingPage functions completed successfully.");
    } else {
      console.warn(
        "Some functions did not complete successfully:",
        failed.map(([fn]) => fn)
      );
    }
  }

  handleServiceBodyClasses() {
    try {
      const enabled = Object.entries(this.config.services)
        .filter(([, isEnabled]) => isEnabled)
        .map(([svc]) => svc);
      if (enabled.length) {
        document.body.classList.add(`svcs-${enabled.join("-")}`);
        return true;
      }
      return false;
    } catch (e) {
      console.error("handleServiceBodyClasses failed", e);
      return false;
    }
  }

  handleModuleClasses() {
    try {
      let success = false;
      document.querySelectorAll("[data-module-type]").forEach((moduleType) => {
        const type = moduleType.getAttribute("data-module-type");
        const parent = moduleType.closest('[id^="page-block"]');
        const elem = moduleType.closest('[id^="element"]');

        if (parent) {
          parent.classList.add(`module--${type}`);

          // Check the next sibling for a variant
          const next = moduleType.nextElementSibling;
          if (next && next.hasAttribute("data-module-variant")) {
            const variant = next.getAttribute("data-module-variant");
            parent.classList.add(`${type}--${variant}`);
          }
        }

        if (elem) elem.remove();
        success = true;
      });
      return success;
    } catch (e) {
      console.error("handleModuleClasses failed", e);
      return false;
    }
  }

  setFormValues() {
    try {
      const branchInputs = document.querySelectorAll('form input[name="Branch ID"]');
      const svcInputs = document.querySelectorAll('form input[name="Service Line"]');
      if (!branchInputs.length && !svcInputs.length) return false;

      branchInputs.forEach((input) => input.setAttribute("value", this.config.branchId));
      svcInputs.forEach((input) => input.setAttribute("value", this.config.svcLine));
      return true;
    } catch (e) {
      console.error("setFormValues failed", e);
      return false;
    }
  }

  handleFormContainer() {
    try {
      const formModule = document.querySelector(".module--lead-form");
      if (!formModule) return false;

      const formElement = formModule.querySelector(".multistep-form");
      if (!formElement) return false;

      let parent = formElement.parentElement;
      while (parent && (!parent.id || !parent.id.startsWith("element-"))) {
        parent = parent.parentElement;
      }

      if (!parent) return false;

      parent.classList.add("form-container");

      return parent.classList.contains("form-container");
    } catch (e) {
      console.error("handleFormContainer failed", e);
      return false;
    }
  }

  replacePhoneCTAs() {
    try {
      const phone = this.config.phone;
      const spans = document.querySelectorAll('span[data-cta="phone"]');
      if (!spans.length) return false;

      spans.forEach((span) => {
        span.textContent = phone;

        const link = span.closest("a");
        if (link) {
          link.setAttribute("href", `tel:${phone}`);
        }
      });

      return true;
    } catch (e) {
      console.error("replacePhoneCTAs failed", e);
      return false;
    }
  }

  replacePlaceholders() {
    try {
      const replacements = {
        Brand: this.config.brand,
        Location: this.config.locRegion,
        SVC: this.config.services.svcString,
        Lead: this.config.services.leadString,
      };
      const walker = document.createTreeWalker(document.body, NodeFilter.SHOW_TEXT);
      const nodes = [];

      let node;
      while ((node = walker.nextNode())) {
        if (/\[.*\]/.test(node.nodeValue)) nodes.push(node);
      }

      for (const node of nodes) {
        let newText = node.nodeValue;
        for (const [key, val] of Object.entries(replacements)) {
          newText = newText.replaceAll(`[${key}]`, val);
        }
        const span = document.createElement("span");
        span.innerHTML = newText;
        node.parentNode.replaceChild(span, node);
      }
      return true;
    } catch (e) {
      console.error("replacePlaceholders failed", e);
      return false;
    }
  }

  handleFinanceModule() {
    try {
      const block = document.querySelector(".module--financing");
      const nav = document.querySelectorAll(".nav-menu__item--financing");

      if (!this.config.hasFinance) {
        let success = false;

        if (block) {
          block.style.display = "none";
          if (block.style.display === "none") {
            success = true;
          }
        }

        if (nav && nav.length > 0) {
          nav.forEach((el) => {
            el.style.display = "none";
          });

          if ([...nav].every((el) => el.style.display === "none")) {
            success = true;
          }
        }

        return success;
      } else {
        if (this.config.hasFinance) {
          let success = false;

          if (block && nav && nav.length > 0) {
            success = true;
          }

          return success;
        }
      }
    } catch (e) {
      console.error("handleFinanceModule failed", e);
      return false;
    }
  }

  handleFDICLogo() {
    try {
      const footerModule = document.querySelector(".module--footer");
      if (!footerModule) return false;

      const logoImage = footerModule.querySelector('img[data-src*="fdic.png"]');
      if (!logoImage) return false;

      let parent = logoImage.parentElement;
      while (parent && (!parent.id || !parent.id.startsWith("element-"))) {
        parent = parent.parentElement;
      }

      if (!parent) return false;

      if (this.config.hasFDICLogo) {
        parent.classList.add("fdic-logo");
        return true;
      } else {
        parent.remove();
        return true;
      }
    } catch (e) {
      console.error("handleFDICLogo failed", e);
      return false;
    }
  }

  injectReviews() {
    try {
      const loc = this.config.locState ? `${this.config.locRegion}, ${this.config.locState}` : this.config.locRegion;
      let found = false;

      document.querySelectorAll(".review__body").forEach((review) => {
        const key = review.getAttribute("data-review-num");
        if (this.reviews[key]) {
          review.querySelector(".review-txt").textContent = this.reviews[key].text;
          found = true;
        }
      });

      document.querySelectorAll(".review__location").forEach((location) => {
        const key = location.previousElementSibling.getAttribute("data-review-num");
        if (this.reviews[key]) {
          location.querySelector(".review-loc").textContent = loc;
          found = true;
        }
      });

      return found;
    } catch (e) {
      console.error("injectReviews failed", e);
      return false;
    }
  }

  injectCoupons() {
    try {
      const container = document.querySelector(".coupons");
      const template = document.querySelector("#coupon-template");
      const disclaimer = document.querySelector("#disclaimer");

      if (!container || !template || !disclaimer || !Object.keys(this.coupons).length) return false;

      Object.keys(this.coupons)
        .sort()
        .forEach((key, idx) => {
          const n = idx + 1;
          const data = this.coupons[key];
          const clone = template.content.cloneNode(true);
          const div = clone.querySelector(".coupon");
          div.classList.remove("coupon-1");
          div.classList.add(`coupon-${n}`);

          clone.querySelector(".coupon__price").innerHTML = data.head;

          const titleEl = clone.querySelector(".coupon__title");
          titleEl.innerHTML = data.subhead;
          const span = document.createElement("span");
          span.className = `lgl${n}`;
          span.textContent = "*".repeat(n);
          titleEl.appendChild(span);

          clone.querySelector(".coupon__desc").innerHTML = data.cta;
          clone.querySelector(".coupon__legal").innerHTML = data.legal;

          const dateMatch = data.legal.match(/Offer expires (\d{2}\/\d{2}\/\d{4})/);
          const expDate = dateMatch ? dateMatch[1] : "xx/xx/xxxx";

          const legalDiv = document.createElement("div");
          legalDiv.className = "legal";
          legalDiv.innerHTML = `
            <p>
              <span class="lgl${n}">${"*".repeat(n)}</span>
              <span class="lgl-txt${n}"></span>
              Offer expires <span class="expire-date-string">${expDate}</span>.
            </p>
          `;
          disclaimer.appendChild(legalDiv);
          container.appendChild(clone);
        });

      document.querySelectorAll(".coupon").forEach((coupon) => {
        const number = coupon.classList[1].split("-")[1];
        const txtEl = document.querySelector(`.lgl-txt${number}`);
        const legal = coupon.querySelector(".coupon__legal");
        if (txtEl && legal) {
          txtEl.textContent = legal.textContent.trim();
          legal.remove();
        }
      });

      // Update popup
      const popup = document.querySelector(".popup-content__promo-head p");
      const popupSub = document.querySelector(".popup-content__promo-subhead .subhead-text");
      if (this.coupons.c1 && popup && popupSub) {
        popup.innerHTML = this.coupons.c1.head;
        popupSub.innerHTML = this.coupons.c1.subhead;
      }

      return true;
    } catch (e) {
      console.error("injectCoupons failed", e);
      return false;
    }
  }

  setupTooltips() {
    try {
      const tooltipPairs = document.querySelectorAll("[data-tooltip-group]");
      if (!tooltipPairs.length) return false;

      tooltipPairs.forEach((group) => {
        const btn = group.querySelector(".tooltip-toggle");
        const body = group.querySelector(".rtb__tooltip, .svc-trust__tooltip");
        const close = group.querySelector(".icon-tooltip-close");
        if (!btn || !body || !close) return;

        btn.addEventListener("click", (e) => {
          e.stopPropagation();
          body.classList.add("active");
        });

        close.addEventListener("click", () => {
          body.classList.remove("active");
        });

        body.dataset.tooltip = "true";
      });

      document.addEventListener("click", (e) => {
        document.querySelectorAll("[data-tooltip='true'].active").forEach((tooltip) => {
          const btn = tooltip.closest("[data-tooltip-group]")?.querySelector(".tooltip-toggle");
          if (!tooltip.contains(e.target) && e.target !== btn) {
            tooltip.classList.remove("active");
          }
        });
      });

      return true;
    } catch (e) {
      console.error("setupTooltips failed", e);
      return false;
    }
  }

  setupMenuToggle() {
    try {
      const menu = document.querySelector("#element-215 .jump-nav-mob");
      const triggers = document.querySelectorAll(".trigger-link");
      if (!menu || !triggers.length) return false;

      let ignoreNext = false;

      window.addEventListener("scroll", () => {
        if (ignoreNext) {
          ignoreNext = false;
          return;
        }
        if (menu.dataset.state === "open") {
          triggers.forEach((t) => t.setAttribute("data-state", "inactive"));
          menu.dataset.state = "closed";
          document.body.classList.remove("menu-open");
          document.querySelector(".module--header")?.classList.remove("menu-open");
        }
      });

      triggers.forEach((trigger) => {
        trigger.addEventListener("click", (e) => {
          e.preventDefault();
          ignoreNext = true;
          const isOpen = menu.dataset.state === "open";
          menu.dataset.state = isOpen ? "closed" : "open";
          document.body.classList.toggle("menu-open", !isOpen);
          document.querySelector(".module--header")?.classList.toggle("menu-open", !isOpen);
          trigger.setAttribute("data-state", isOpen ? "inactive" : "active");
        });
      });

      return true;
    } catch (e) {
      console.error("setupMenuToggle failed", e);
      return false;
    }
  }

  equalizeCouponHeights() {
    try {
      const setHeights = () => {
        if (window.innerWidth <= 768) return;

        const coupons = document.querySelectorAll(".coupon");
        let maxTitle = 0;
        let maxDesc = 0;

        coupons.forEach((c) => {
          const title = c.querySelector(".coupon__title");
          const desc = c.querySelector(".coupon__desc");
          if (title) {
            title.style.height = "auto";
            maxTitle = Math.max(maxTitle, title.offsetHeight);
          }
          if (desc) {
            desc.style.height = "auto";
            maxDesc = Math.max(maxDesc, desc.offsetHeight);
          }
        });

        coupons.forEach((c) => {
          c.querySelector(".coupon__title").style.height = `${maxTitle}px`;
          c.querySelector(".coupon__desc").style.height = `${maxDesc}px`;
        });
      };

      setHeights();
      window.addEventListener("resize", setHeights);
      return true;
    } catch (e) {
      console.error("equalizeCouponHeights failed", e);
      return false;
    }
  }

  updateExpirationDates() {
    try {
      const now = new Date();
      const lastDay = new Date(now.getFullYear(), now.getMonth() + 1, 0);
      const dateStr = `${String(lastDay.getMonth() + 1).padStart(2, "0")}/${String(lastDay.getDate()).padStart(2, "0")}/${lastDay.getFullYear()}`;

      document.querySelectorAll(".expire-date-string").forEach((el) => {
        el.textContent = dateStr;
      });

      return true;
    } catch (e) {
      console.error("updateExpirationDates failed", e);
      return false;
    }
  }
}